// Layer Picker behavior
/*global define, console */

define(["lib/Zoot", "src/math/Vec2"],
function (Z, v2) {
	"use strict";

	var	kKeyModeOnHold = 0,
		kKeyModeOnTap = 1;
	var	kOutOfRangeClamp = 0,
		kOutOfRangeLoop = 1;

	function showOnlyLayer(args, layerIndexToShow) {
		var i = 0;
		args.stageLayer.forEachDirectChildLayer(function (lay) {
			if (i++ === layerIndexToShow) {
				lay.trigger();	// Previously set to -1 to lower priority, but it didn't work so setting back to blank
				return;
			}
		});
	}

	function countLayers(args) {
		var i = 0;
		args.stageLayer.forEachDirectChildLayer(function () {
			++i;
		});
		return i;
	}

	function clampToNumLayers(args, pickedLayer, numLayers) {
		var maxLayers = numLayers - 1, minLayers = 0;
		if (args.getParam("outOfRangeMode") === kOutOfRangeClamp) {
			if (pickedLayer > maxLayers) {
				pickedLayer = maxLayers;
			}
			else if (pickedLayer < minLayers) {
				pickedLayer = minLayers;
			}
		}
		else {
			// assume kOutOfRangeLoop
			pickedLayer = pickedLayer % numLayers;
			if (pickedLayer < minLayers) {
				pickedLayer += numLayers;
			}
		}
		return pickedLayer;
	}
	
	function getLayerIndexOffset(args) {
		return args.getParam("layerIndex") - 1;
	}

	function getLayerPercentageOffset(args, numLayers) {
		var pctValue = args.getParam("layerPct"),
			rangeParam = args.getParam("outOfRangeMode");
		
		if (rangeParam === kOutOfRangeLoop && pctValue < 0) {
			pctValue = 100 - Math.abs(pctValue);
		}
		
		return Math.round(pctValue / 100.0 * (numLayers - 1));
	}

	function getLayerAudioOffset(args, numLayers) {
		var	audioOffset = 0,
			audioAmplitude, audioSensitivity;
		
		// strange that we have to check something related to Viseme, but apparently
		// that's the only way to see if the mic is on (Audio/Amplitude freezes at last
		// value when it's off)
		if (args.getParamEventValue("audioInput", "Viseme/InputEnabled")) {
			args.setEventGraphParamRecordingValid("audioInput");

			audioAmplitude = args.getParamEventValue("audioInput", "Audio/Amplitude");
			audioSensitivity = args.getParam("audioFactor") * 0.01;

			audioOffset = audioAmplitude * audioSensitivity * (numLayers - 1);
			//console.logToUser("  audio=" + audioOffset + " (amplitude=" + audioAmplitude + ", sensitivity=" + audioSensitivity + ")");
		}
		
		return audioOffset;
	}

	function getLayerKeyOffset(self, args, numLayers) {
		var	keyOffset,
			kc = Z.keyCodes, leftArrowDown, upArrowDown, rightArrowDown, downArrowDown,
			bKeyChangeOnHold = (args.getParam("keyMode") === kKeyModeOnHold);
		
		keyOffset = self.totalKeyOffset;
		leftArrowDown = args.getParamEventValue("keyInput", kc.getKeyGraphId(kc.leftKey), null, 0);
		upArrowDown = args.getParamEventValue("keyInput", kc.getKeyGraphId(kc.upKey), null, 0);
		rightArrowDown = args.getParamEventValue("keyInput", kc.getKeyGraphId(kc.rightKey), null, 0);
		downArrowDown = args.getParamEventValue("keyInput", kc.getKeyGraphId(kc.downKey), null, 0);
		//console.logToUser("  left=" + leftArrowDown + ", up=" + upArrowDown + ", right=" + rightArrowDown + ", down=" + downArrowDown);

		// check if an Arrow key is down, and also if only interested in taps vs. holds
		if (leftArrowDown || upArrowDown || rightArrowDown || downArrowDown) {
			if (!bKeyChangeOnHold && self.bKeyPreviouslyDown) {
				// skip, because we're only interested in taps
			}
			else {
				self.bKeysValid = true;
				if (leftArrowDown) {
					keyOffset--;
				} else if (upArrowDown) {
					keyOffset = 0;
				} else if (downArrowDown) {
					keyOffset = numLayers - 1;
				} else {
					// assume rightArrowDown
					keyOffset++;
				}
				self.bKeyPreviouslyDown = true;
			}
		}
		else {
			self.bKeyPreviouslyDown = false;				
		}
		
		if (self.bKeysValid) {
			args.setEventGraphParamRecordingValid("keyInput");
		}
		var clampInputB = false;
		if (clampInputB) {
			keyOffset = clampToNumLayers(args, keyOffset, numLayers);
		}
		self.totalKeyOffset = keyOffset;
		return keyOffset;
	}

	function getLayerMouseTouchOffset(self, args, numLayers) {
		var	mouseTouchOffset,
			leftDownB, mouseTouchDownOffset, mousePosition0, mouseTouchStrength,
			touchEvents, touchId, touchIdKey;
		
		mouseTouchOffset = self.totalMouseTouchOffset;
		mouseTouchDownOffset = self.currentMouseTouchOffset;
		leftDownB = args.getParamEventValue("mouseTouchInput", "Mouse/Down/Left", null, 0, true);
		mousePosition0 = args.getParamEventValue("mouseTouchInput", "Mouse/Position", null, [0,0], true);

		touchEvents = args.getParamEventValue("mouseTouchInput", "Touch/IndexToID/Count", null, 0, true);
		if (touchEvents) {
			touchId = args.getParamEventValue("mouseTouchInput", "Touch/IndexToID/0", null, 0, true);
			touchIdKey = "Touch/ID/" + touchId;
			leftDownB = args.getParamEventValue("mouseTouchInput", touchIdKey + "/Down", null, 0, true);
			if (leftDownB) {
				mousePosition0 = args.getParamEventValue("mouseTouchInput", touchIdKey + "/Position", null, [0,0], true);
			}
			//console.logToUser("  touch: pos=" + mousePosition0 + ", vec=" + touchVec);
		}

		if (leftDownB && mousePosition0) {
			self.bMouseOrTouchValid = true;

			mouseTouchStrength = args.getParam("mouseTouchFactor") * 0.0001;

			if (self.mouseFirstDownPos === undefined) {
				self.mouseFirstDownPos = mousePosition0;
			}
			mouseTouchDownOffset = v2.subtract(mousePosition0, self.mouseFirstDownPos)[0] * mouseTouchStrength;
			//console.logToUser("  totalDist=" + mouseTouchOffset + ", currDist=" + mouseTouchDownOffset + ", firstDown=" + self.mouseFirstDownPos + ", pos=" + mousePosition0);
		}
		else {
			// reset total offsets and positions on mouse up
			mouseTouchOffset += mouseTouchDownOffset;
			mouseTouchDownOffset = 0;
			//console.logToUser("  mouseUp: mouseTouchOffset=" + mouseTouchOffset + ", mouseTouchDownOffset=" + mouseTouchDownOffset);

			self.mouseFirstDownPos = undefined;
			self.totalMouseTouchOffset = mouseTouchOffset;
		}
		mouseTouchOffset = self.totalMouseTouchOffset + mouseTouchDownOffset;

		if (self.bMouseOrTouchValid) {
			args.setEventGraphParamRecordingValid("mouseTouchInput");
		}
	
		var clampInputB = false;
		if (clampInputB) {
			mouseTouchOffset = clampToNumLayers(args, mouseTouchOffset, numLayers);
		}

		self.currentMouseTouchOffset = mouseTouchDownOffset;
		return mouseTouchOffset;
	}

	function getLayerCameraOffset(self, args) {
		var	camOffset = 0, camStrength;
		
        if (args.getParamEventValue("cameraInput", "Head/InputEnabled", null, false)) {
            args.setEventGraphParamRecordingValid("cameraInput");
            
			camStrength = args.getParam("camFactor") * 0.1;
			camOffset = args.getParamEventValue("cameraInput", "Head/DX", null, "Head/DX") * camStrength;
			//console.logToUser(" cam: dx=" + camOffset);
		}
		
		return camOffset;
	}
	
	return {
		about:			"$$$/private/animal/Behavior/LayerPicker/About=Layer Picker, (c) 2016-2017 Adobe. All rights reserved.",
		description:	"$$$/animal/Behavior/LayerPicker/Desc=Trigger a specific layer in a puppet or group",
		uiName:  		"$$$/animal/Behavior/LayerPicker/UIName=Layer Picker",
		defaultArmedForRecordOn: true,

		defineParams : function () {
			return [
				{
					id: "audioInput", type: "eventGraph",
				 	uiName: "$$$/animal/Behavior/LayerPicker/Parameter/audioInput=Audio Input",
				 	inputKeysArray: ["Viseme/InputEnabled", "Audio/Amplitude"],
					uiToolTip: "$$$/animal/Behavior/LayerPicker/Parameter/audioInput/tooltip=Offset the index of the layer to pick by talking into the microphone; microphone must be enabled",
				 	defaultArmedForRecordOn: false
				},
				{
					id: "keyInput", type: "eventGraph", 
					uiName: "$$$/Behavior/LayerPicker/Parameter/keyboardInput=Keyboard Input",
					inputKeysArray: ["Keyboard/"],
					outputKeyTraits: {
						takeGroupsArray: [
							{
								id: "LayerOffset/*"
							}
						]
					},
					supportsBlending: true,
					uiToolTip: "$$$/Behavior/LayerPicker/Parameter/keyboardInput/tooltip=Offset the index of the layer to pick by pressing the Left Arrow key (lower-numbered layers) or Right Arrow key (higher-numbered layers); Up Arrow picks the first layer, Down Arrow the last layer",
					defaultArmedForRecordOn: false
				},
				{
					id: "mouseTouchInput", type: "eventGraph", 
					uiName: "$$$/Behavior/LayerPicker/Parameter/mouseTouchInput=Mouse & Touch Input",
					inputKeysArray: ["Mouse/", "Touch/"],
					outputKeyTraits: {
						takeGroupsArray: [
							{
								id: "LayerOffset/*"
							}
						]
					},
					supportsBlending: true,
					uiToolTip: "$$$/Behavior/LayerPicker/Parameter/mouseTouchInput/tooltip=Offset the index of the layer to pick by dragging with the mouse or on a touch-enabled display, relative to the center of the scene",
					defaultArmedForRecordOn: false
				},
				{
                    id: "cameraInput", type: "eventGraph", 
                    uiName: "$$$/private/animal/Behavior/LayerPicker/Parameter/cameraInput=Camera Input (Head X Position)",
                    inputKeysArray: ["Head/"],
                    disarmIfAllInputKeysDisabledArray: ["Head/"], // Used to determine whether to disarm if dependent hardware is disabled.
                    uiToolTip: "$$$/private/animal/Behavior/LayerPicker/param/cameraInput/tooltip=Offset the index of the layer to pick by moving your face horizontally in front of your webcam",
                    defaultArmedForRecordOn: false, supportsBlending: true,
                    hidden: true
                },
				{
					id: "layerIndex", type: "slider", uiName: "$$$/animal/Behavior/LayerPicker/Parameter/layerIndex=Index Offset", 
					min: 1, max: 1000, dephault: 1,
					uiToolTip: "$$$/animal/behavior/LayerPicker/Parameter/layerIndex/tooltip=Offset the index of the layer to pick by choosing a number (minimum of 1)"
				},
				{
					id: "layerPct", type: "slider", uiName: "$$$/animal/Behavior/LayerPicker/Parameter/layerPct=Percentage Offset", 
					min: -100, max: 100, dephault: 0, uiUnits: "%",
					uiToolTip: "$$$/animal/behavior/LayerPicker/Parameter/layerPct/tooltip=Offset the index of the layer to pick by choosing a percentage of the range of layers"
				},
				{
					id: "audioFactor", type: "slider",
				 	uiName: "$$$/animal/Behavior/LayerPicker/Parameter/audioSensitivity=Audio Sensitivity",
				 	uiUnits: "%", dephault: 100,
					uiToolTip: "$$$/animal/Behavior/LayerPicker/Parameter/audioSensitivity/tooltip=How much influence audio volume has on the layer to pick",
				},
				{
					id: "mouseTouchFactor", type: "slider",
				 	uiName: "$$$/animal/Behavior/LayerPicker/Parameter/mouseTouchFactor=Mouse & Touch Strength",
				 	uiUnits: "%", dephault: 100,
					uiToolTip: "$$$/animal/Behavior/LayerPicker/Parameter/mouseTouchFactor/tooltip=How far dragging horizontally by mouse or touch has on the layer to pick",
					hideRecordButton: true,
				},
				{
					id: "camFactor", type: "slider",
				 	uiName: "$$$/private/animal/Behavior/LayerPicker/Parameter/camFactor=Camera Strength",
				 	uiUnits: "%", dephault: 100,
					uiToolTip: "$$$/private/animal/Behavior/LayerPicker/Parameter/camFactor/tooltip=How far shifting your head horizontally in front of your webcam has on the layer to pick",
                    hidden: true
				},
				{
					id: "keyMode", type: "enum",
					uiName: "$$$/animal/Behavior/LayerPicker/Parameter/keyMode=Keyboard Advance",
					items: [
						{ id: kKeyModeOnTap,	uiName: "$$$/animal/Behavior/LayerPicker/Parameter/keyMode/onTap=On Tap" },
						{ id: kKeyModeOnHold,	uiName: "$$$/animal/Behavior/LayerPicker/Parameter/keyMode/onHold=On Hold" },
					],
					dephault: kKeyModeOnTap,
					uiToolTip: "$$$/animal/Behavior/LayerPicker/Parameter/keyMode/tooltip=How to use the Arrow key (while holding it down or tapping) to change the layer index",
					hideRecordButton: true,
				},
				{
					id: "outOfRangeMode", type: "enum",
					uiName: "$$$/animal/Behavior/LayerPicker/Parameter/outOfRangeMode=Out of Range",
					items: [
						{ id: kOutOfRangeClamp,	uiName: "$$$/animal/Behavior/LayerPicker/Parameter/outOfRangeMode/clampToRange=Limit to First/Last" },
						{ id: kOutOfRangeLoop,	uiName: "$$$/animal/Behavior/LayerPicker/Parameter/outOfRangeMode/loopRange=Loop" },
					],
					dephault: kOutOfRangeClamp,
					uiToolTip: "$$$/animal/Behavior/LayerPicker/Parameter/outOfRangeMode/tooltip=How to pick layers when the index is less than 1 or greater than the number of layers",
				},
			];
		},
		
		onCreateBackStageBehavior : function (/*self, args*/) {
			//console.logToUser("onCreateBackStageBehavior");
			return { order: 0, importance : 0.0 };	// must come before anything that can be triggered
		},

		onCreateStageBehavior : function (self, args) {
			//console.logToUser("onCreateStageBehavior");			
			var bHideSiblings = true;
			args.stageLayer.forEachDirectChildLayer(function (lay) {
				lay.setTriggerable(bHideSiblings);
			});

			this.onResetRehearsalData(self);
		},

		onResetRehearsalData : function (self) {
			self.totalKeyOffset = 0;			// cumulative layer index (offset) based on key presses
			self.bKeyPreviouslyDown = false;	// indicates if no Arrow key was previously held down
			self.bKeysValid = false;			// indicates the key interactions are valid
			self.totalMouseTouchOffset = 0;		// cumulative layer index (offset) based on all mouse and touch dragging
			self.currentMouseTouchOffset = 0;	// cumulative layer index (offset) based on current (mouse-down) mouse and touch dragging
			self.mouseFirstDownPos = undefined;	// indicates the position when the mouse was first held down; undefined if mouse up
			self.bMouseOrTouchValid = false;	// indicates the mouse or touch interactions are valid
		},

		onFilterLiveInputs : function (self, args) { // method on behavior that is attached to a puppet, only onstage
			var paramOutputKey, numLayers = countLayers(args), offsetIndex;
			if (args.isParamEventLive("keyInput")) {
				paramOutputKey = args.getParamEventOutputKey("keyInput") + "LayerOffset/Index";
				offsetIndex = getLayerKeyOffset(self, args, numLayers);

				// Publish the spring/latch values.
				if (self.bKeysValid || args.eventGraph.hasGraph(paramOutputKey)) {
					args.eventGraph.publish1D(paramOutputKey, args.currentTime, offsetIndex, true);			
				}
			}
			if (args.isParamEventLive("mouseTouchInput")) {
				paramOutputKey = args.getParamEventOutputKey("mouseTouchInput") + "LayerOffset/Index";
				offsetIndex = getLayerMouseTouchOffset(self, args, numLayers);
				// Publish the spring/latch values.
				if (self.bMouseOrTouchValid || args.eventGraph.hasGraph(paramOutputKey)) {
					args.eventGraph.publish1D(paramOutputKey, args.currentTime, offsetIndex, true);			
				}
			}
		},

		onAnimate : function (self, args) {
			var	numLayers = countLayers(args),
				pickedLayer, pickedLayerWithNoKeyOffset,
				indexOffset, pctOffset, audioOffset, keyOffset, mouseTouchOffset, camOffset;

			if (numLayers === 0) {
				return;
			}

			// get each picker's offset (all zero based);
			indexOffset = getLayerIndexOffset(args);
			pctOffset = getLayerPercentageOffset(args, numLayers);
			audioOffset = getLayerAudioOffset(args, numLayers);
			
			keyOffset = args.getParamEventValue("keyInput", args.getParamEventOutputKey("keyInput") + "LayerOffset/Index", null, 0) || 0;
			mouseTouchOffset = args.getParamEventValue("mouseTouchInput", args.getParamEventOutputKey("mouseTouchInput") + "LayerOffset/Index", null, 0) || 0;
			//console.log("mouseTouchInput's offsetIndex = " + mouseTouchOffset);
			camOffset = getLayerCameraOffset(self, args);

			// sum all offsets, then clamp or loop to range of layers
			pickedLayer = indexOffset + pctOffset + audioOffset + keyOffset + mouseTouchOffset + camOffset;
			pickedLayerWithNoKeyOffset = pickedLayer - keyOffset;
			//console.logToUser("  pickedLayer=" + pickedLayer + ", key=" + keyOffset + ", pickedLayer-key=" + (pickedLayer - keyOffset));
			
			pickedLayer = Math.round(pickedLayer);	// round before doing any clamping
			pickedLayer = clampToNumLayers(args, pickedLayer, numLayers);
			//console.logToUser("pickedLayer=" + pickedLayer + " (index=" + indexOffset + ", pct=" + pctOffset + ", audio=" + audioOffset + ", key=" + keyOffset + ", mouse=" + mouseTouchOffset + "), cam=" + camOffset);

			showOnlyLayer(args, pickedLayer);
		}
	}; // end of object being returned
});
